package nemosofts.voxradio.widget;

import android.app.PendingIntent;
import android.appwidget.AppWidgetManager;
import android.appwidget.AppWidgetProvider;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.os.Build;
import android.widget.RemoteViews;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.OptIn;
import androidx.media3.common.util.UnstableApi;
import androidx.nemosofts.utils.NetworkUtils;

import java.util.ArrayList;
import java.util.List;

import nemosofts.voxradio.R;
import nemosofts.voxradio.activity.NSoftsPlayerActivity;
import nemosofts.voxradio.activity.PlayerService;
import nemosofts.voxradio.callback.Callback;
import nemosofts.voxradio.callback.Method;
import nemosofts.voxradio.executor.LoadRadio;
import nemosofts.voxradio.item.ItemRadio;
import nemosofts.voxradio.interfaces.RadioListener;
import nemosofts.voxradio.utils.helper.Helper;
import nemosofts.voxradio.utils.helper.SPHelper;

public class PlayerWidgetProvider extends AppWidgetProvider {

    private static final String ACTION_WIDGET_PLAY = "widget.action.PLAY";
    private static final String ACTION_WIDGET_NEXT = "widget.action.NEXT";
    private static final String ACTION_WIDGET_PREVIOUS = "widget.action.PREVIOUS";
    private static boolean isLoadingLatest = false;

    @Override
    public void onUpdate(@NonNull Context context, @NonNull AppWidgetManager appWidgetManager, @NonNull int[] appWidgetIds) {
        super.onUpdate(context, appWidgetManager, appWidgetIds);
        for (int appWidgetId : appWidgetIds) {
            updateWidget(context, appWidgetManager, appWidgetId);
        }
    }

    @OptIn(markerClass = UnstableApi.class)
    @Override
    public void onReceive(@NonNull Context context, @NonNull Intent intent) {
        super.onReceive(context, intent);
        String action = intent.getAction();
        if (AppWidgetManager.ACTION_APPWIDGET_UPDATE.equals(action)) {
            notifyWidgets(context);
            return;
        }
        if (ACTION_WIDGET_PLAY.equals(action)) {
            handlePlay(context);
            return;
        }
        if (ACTION_WIDGET_NEXT.equals(action)) {
            handleSkip(context, PlayerService.ACTION_NEXT);
            return;
        }
        if (ACTION_WIDGET_PREVIOUS.equals(action)) {
            handleSkip(context, PlayerService.ACTION_PREVIOUS);
        }
    }

    public static void notifyWidgets(@NonNull Context context) {
        AppWidgetManager appWidgetManager = AppWidgetManager.getInstance(context);
        ComponentName thisWidget = new ComponentName(context, PlayerWidgetProvider.class);
        int[] appWidgetIds = appWidgetManager.getAppWidgetIds(thisWidget);
        for (int appWidgetId : appWidgetIds) {
            updateWidget(context, appWidgetManager, appWidgetId);
        }
    }

    @OptIn(markerClass = UnstableApi.class)
    private static void updateWidget(@NonNull Context context, @NonNull AppWidgetManager appWidgetManager, int appWidgetId) {
        RemoteViews views = new RemoteViews(context.getPackageName(), R.layout.widget_player);

        ItemRadio current = getCurrentRadio();
        if (current != null) {
            views.setTextViewText(R.id.widget_station_title, current.getRadioTitle());
            views.setTextViewText(R.id.widget_station_subtitle, current.getCategoryName());
        } else {
            views.setTextViewText(R.id.widget_station_title, context.getString(R.string.app_name));
            views.setTextViewText(R.id.widget_station_subtitle, context.getString(R.string.made_with_in));
        }

        boolean isPlaying = PlayerService.getIsPlaying();
        views.setImageViewResource(R.id.widget_play_pause, isPlaying ? R.drawable.ic_pause : R.drawable.ic_play);

        Bitmap cover = PlayerService.getCurrentBitmap();
        if (cover != null) {
            views.setImageViewBitmap(R.id.widget_cover, cover);
        } else {
            views.setImageViewResource(R.id.widget_cover, R.drawable.ic_radio);
        }

        PendingIntent openAppIntent = PendingIntent.getActivity(
                context,
                0,
                new Intent(context, NSoftsPlayerActivity.class),
                PendingIntent.FLAG_IMMUTABLE
        );
        views.setOnClickPendingIntent(R.id.widget_root, openAppIntent);
        views.setOnClickPendingIntent(R.id.widget_station_title, openAppIntent);
        views.setOnClickPendingIntent(R.id.widget_station_subtitle, openAppIntent);
        views.setOnClickPendingIntent(R.id.widget_cover, openAppIntent);

        views.setOnClickPendingIntent(R.id.widget_play_pause, buildBroadcastPendingIntent(context, ACTION_WIDGET_PLAY, 100));
        views.setOnClickPendingIntent(R.id.widget_next, buildBroadcastPendingIntent(context, ACTION_WIDGET_NEXT, 101));
        views.setOnClickPendingIntent(R.id.widget_prev, buildBroadcastPendingIntent(context, ACTION_WIDGET_PREVIOUS, 102));

        appWidgetManager.updateAppWidget(appWidgetId, views);
    }

    private static PendingIntent buildBroadcastPendingIntent(@NonNull Context context, @NonNull String action, int requestCode) {
        Intent intent = new Intent(context, PlayerWidgetProvider.class);
        intent.setAction(action);
        return PendingIntent.getBroadcast(context, requestCode, intent, PendingIntent.FLAG_IMMUTABLE);
    }

    private static ItemRadio getCurrentRadio() {
        List<ItemRadio> playlist = Callback.getArrayListPlay();
        int pos = Callback.getPlayPos();
        if (playlist.isEmpty() || pos < 0 || pos >= playlist.size()) {
            return null;
        }
        return playlist.get(pos);
    }

    @OptIn(markerClass = UnstableApi.class)
    private void handlePlay(@NonNull Context context) {
        if (!NetworkUtils.isConnected(context)) {
            Toast.makeText(context, context.getString(R.string.err_internet_not_connected), Toast.LENGTH_SHORT).show();
            return;
        }
        if (Callback.getArrayListPlay().isEmpty()) {
            loadLatestAndPlay(context);
            return;
        }
        startServiceAction(context, PlayerService.ACTION_TOGGLE);
    }

    private void handleSkip(@NonNull Context context, @NonNull String action) {
        if (!NetworkUtils.isConnected(context)) {
            Toast.makeText(context, context.getString(R.string.err_internet_not_connected), Toast.LENGTH_SHORT).show();
            return;
        }
        if (Callback.getArrayListPlay().isEmpty()) {
            loadLatestAndPlay(context);
            return;
        }
        startServiceAction(context, action);
    }

    private void loadLatestAndPlay(@NonNull Context context) {
        if (isLoadingLatest) {
            Toast.makeText(context, R.string.loading, Toast.LENGTH_SHORT).show();
            return;
        }
        isLoadingLatest = true;
        setLoadingState(context, true);
        Helper helper = new Helper(context);
        SPHelper spHelper = new SPHelper(context);
        new LoadRadio(new RadioListener() {
            @Override
            public void onStart() {
                // handled by setLoadingState
            }

            @OptIn(markerClass = UnstableApi.class)
            @Override
            public void onEnd(String success, String verifyStatus, String message, ArrayList<ItemRadio> arrayListRadio) {
                isLoadingLatest = false;
                setLoadingState(context, false);
                if (!"1".equals(success) || arrayListRadio == null || arrayListRadio.isEmpty()) {
                    Toast.makeText(context, message.isEmpty() ? context.getString(R.string.err_server_not_connected) : message, Toast.LENGTH_SHORT).show();
                    return;
                }
                Callback.getArrayListPlay().clear();
                Callback.setArrayListPlay(arrayListRadio);
                Callback.setPlayPos(0);
                startServiceAction(context, PlayerService.ACTION_PLAY);
                notifyWidgets(context);
            }
        }, helper.getAPIRequest(Method.METHOD_LATEST, 1, "", "", "",
                "", spHelper.getUserId(), "", "", "", "",
                "", "", "", null
        )).execute();
    }

    @OptIn(markerClass = UnstableApi.class)
    private void startServiceAction(@NonNull Context context, @NonNull String action) {
        Intent intent = new Intent(context, PlayerService.class);
        intent.setAction(action);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            context.startForegroundService(intent);
        } else {
            context.startService(intent);
        }
    }

    @OptIn(markerClass = UnstableApi.class)
    private void setLoadingState(@NonNull Context context, boolean loading) {
        AppWidgetManager appWidgetManager = AppWidgetManager.getInstance(context);
        ComponentName thisWidget = new ComponentName(context, PlayerWidgetProvider.class);
        int[] appWidgetIds = appWidgetManager.getAppWidgetIds(thisWidget);
        for (int appWidgetId : appWidgetIds) {
            RemoteViews views = new RemoteViews(context.getPackageName(), R.layout.widget_player);
            if (loading) {
                views.setImageViewResource(R.id.widget_play_pause, R.drawable.ic_refresh);
                views.setBoolean(R.id.widget_play_pause, "setEnabled", false);
                views.setBoolean(R.id.widget_next, "setEnabled", false);
                views.setBoolean(R.id.widget_prev, "setEnabled", false);
            } else {
                views.setImageViewResource(R.id.widget_play_pause, PlayerService.getIsPlaying() ? R.drawable.ic_pause : R.drawable.ic_play);
                views.setBoolean(R.id.widget_play_pause, "setEnabled", true);
                views.setBoolean(R.id.widget_next, "setEnabled", true);
                views.setBoolean(R.id.widget_prev, "setEnabled", true);
            }
            appWidgetManager.updateAppWidget(appWidgetId, views);
        }
    }
}
